import app from 'app';
import {EventEmitter} from 'events';
import SquirrelUpdate from './squirrel-update';

class AutoUpdater extends EventEmitter {
  constructor() {
    super.constructor();

    this.updateUrl = null;
    this.updateVersion = null;
  }

  setFeedUrl(updateUrl) {
    this.updateUrl = updateUrl;
  }

  quitAndInstall() {
    if (SquirrelUpdate.existsSync()) {
      SquirrelUpdate.restart(app, this.updateVersion || app.getVersion());
    }
    else {
      require('auto-updater').quitAndInstall();
    }
  }

  downloadUpdate(callback) {
    SquirrelUpdate.spawnUpdate(['--download', this.updateUrl], (error, stdout) => {
      if (error != null) {
        return callback(error);
      }

      try {
        // Last line of the output is JSON detauls about the releases
        let json = stdout.trim().split('\n').pop();
        let update = JSON.parse(json).releasesToApply.pop();
        callback(null, update);
      }
      catch (error) {
        error.stdout = stdout;
        callback(error);
      }
    });
  }

  installUpdate(callback) {
    SquirrelUpdate.spawnUpdate(['--update', this.updateUrl], callback);
  }

  checkForUpdates() {
    if (this.updateUrl == null) {
      throw new Error('Update URL is not set');
    }

    this.emit('checking-for-update');

    if (!SquirrelUpdate.existsSync()) {
      this.emit('update-not-available');
      return;
    }

    this.downloadUpdate((error, update) => {
      if (error != null) {
        this.emit('update-not-available');
        return;
      }

      if (update == null) {
        this.emit('update-not-available');
        return;
      }

      this.installUpdate(error => {
        if (error != null) {
          this.emit('update-not-available');
          return;
        }

        this.updateVersion = update.version;

        this.emit('update-available');
        this.emit('update-downloaded',
          {},
          update.release,
          update.version,
          new Date(),
          this.updateUrl,
          this.quitAndInstall.bind(this));
      });
    });
  }
}

export default new AutoUpdater;
